import tkinter as tk
from state import Board

class Connect4UI(tk.Frame):

    def __init__(self, player1, player2, master: tk.Tk):
        super().__init__(master)
        self._master = master
        self.pack()
        self._ais = {}
        self._ais[1] = player1
        self._ais[-1] = player2
        self._game_number = 0
        self._board = Board()
        self._done = False

        self._create_widgets()

        self._master.protocol("WM_DELETE_WINDOW", self._on_close)

        self._update()

    def _player(self):
        return self._board.get_next_player()

    def _create_widgets(self):
        self._status_label = tk.Label(self, text="", font=("Helvetica", 16))
        self._status_label.pack(side="top", fill="x", expand=False)

        self._board_frame = tk.Frame(self, border=2, relief="solid")
        self._board_frame.pack(side="top", fill="both", expand=True, padx=20, pady=10)

        self._cells = []
        for x in range(self._board.get_width()):
            column = []
            column_frame = tk.Frame(self._board_frame)
            for y in range(self._board.get_height()):
                cell = tk.Canvas(column_frame, borderwidth=0, highlightthickness=0,
                                width=100, height=100, bg="#2057c6")
    
                cell.pack(side="bottom", fill="both", expand=True)
                cell.pack_propagate(0)
                column.append(cell)

            column_frame.pack(side="left", fill="both", expand=True)
            self._cells.append(column)


        self._button_frame = tk.Frame(self)
        self._button_frame.pack(side="bottom", fill="x", expand=False, padx=10, pady=10)

        self._button_new = tk.Button(self._button_frame, text="Nouveau jeu", command=self._new_game)
        self._button_new.pack(side="left", fill="x", expand=True)

        self._button_quit = tk.Button(self._button_frame, text="Quitter", command=self._on_close)
        self._button_quit.pack(side="left", fill="x", expand=True)

    def _on_close(self):
        self._master.destroy()
    
    def _new_game(self):
        self._game_number += 1
        self._board = Board()
        self._done = False
        self._update()
    
    def _update(self):
        
        free_columns = self._board.free_columns()
        if not free_columns:
            self._status_label["text"] = "Partie nulle"
            self._done = True
        
        if not self._done:
            name = "1" if self._player() == 1 else "2"
            self._status_label["text"] = "Au joueur {} de jouer".format(name)

        over_column = None

        for x, column in enumerate(self._cells):
            free_column = x in free_columns

            for y, cell in enumerate(column):
                cell["bg"] = "#2057c6"
                cell["cursor"] = "arrow"

                for child in cell.winfo_children():
                    child.destroy()

                player = self._board.get_token(x, y)

                cell.unbind("<Button-1>")
                cell.unbind("<Enter>")
                cell.unbind("<Leave>")
                
                if player == 0:
                    color = "#ececec"
                elif player == 1:
                    color = "red"
                else:
                    color = "yellow"

                cell.delete("all")
                cell.create_oval(5, 5, 95, 95, fill=color, outline="black", width=2)

                if free_column and not self._done and self._ais[self._player()] is None:
                    cell.bind("<Button-1>", lambda event, x=x: self._on_play(x))
                    cell.bind("<Enter>", lambda event, x=x: self._highlight_column(x))
                    cell.bind("<Leave>", lambda event, x=x: self._unhighlight_column(x))

                    # Check if cursor is already on the cell
                    cursor_x = self._master.winfo_pointerx()
                    cursor_y = self._master.winfo_pointery()
                    if cell.winfo_containing(cursor_x, cursor_y) == cell:
                        over_column = x
        
        if over_column is not None:
            self._highlight_column(over_column)

        if self._ais[self._player()] is not None and not self._done:
            game = self._game_number
            self.after(50, lambda: self._ai_play(game))
    
    def _ai_play(self, game):
        if game != self._game_number:
            return

        column = self._ais[self._player()](self._board)
        self._on_play(column)

    def _on_play(self, column):
        name = "1" if self._player() == 1 else "2"
        if self._board.play(column):
            self._done = True
            self._status_label["text"] = "Le joueur {} a gagné !".format(name)
        self._update()

    def _highlight_column(self, column):
        for cell in self._cells[column]:
            cell["bg"] = "#5081e2"
            cell["cursor"] = "hand2"
        
    def _unhighlight_column(self, column):
        for cell in self._cells[column]:
            cell["bg"] = "#2057c6"
            cell["cursor"] = "arrow"

def start_ui(player1, player2):
    root = tk.Tk()
    root.title("Puissance 4")
    root.resizable(False, False)
    root.eval('tk::PlaceWindow . center')
    ui = Connect4UI(player1, player2, master=root)
    ui.mainloop()
